/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
  i-net software 1998-2013

*/
using System;
using System.Web;
using System.Text;
using System.Security.Policy;
namespace Inet.Viewer.Data
{
    /// <summary>
    /// Kind of a link (hyperlink, subreport, sorting)
    /// </summary>
    public enum LinkType
    {
        /// <summary>
        /// Link to a subreport
        /// </summary>
        SubreportOnDemand = 0,
        /// <summary>
        /// Link to a URL
        /// </summary>
        Hyperlink = 1,
        /// <summary>
        /// Interactive Sorting
        /// </summary>
        InteractiveSorting = 2,
        /// <summary>
        /// No link, only a tooltip (internal link type)
        /// </summary>
        ToolTip = -1,
    }

    /// <summary>
    /// This class represents the interactive features like Hyperlinks, Subreports on demand, or drilldown.
    /// It contains the information about the rectangle with position and the meta information.
    /// </summary>
    internal class PageClip
    {
        private int x, y, width, height;
        private string tabName;
        // for links equals url.toExternalForm()
        private string toolTip;

        /// <summary>
        /// The hyperlink URL 
        /// </summary>
        public Uri Url { get; private set; }

        /// <summary>
        /// The sub-report URL
        /// </summary>
        public string SubReportURL { get; private set; }

        /// <summary>
        /// The link type
        /// </summary>
        public LinkType LinkType { get; private set; }

        /// <summary>
        /// Base constructor
        /// </summary>
        /// <param name="x"> the x position of the clip </param>
        /// <param name="y"> the y position of the clip </param>
        /// <param name="width"> the width of the clip </param>
        /// <param name="height"> the height of the clip </param>
        private PageClip(int x, int y, int width, int height)
        {
            this.x = x;
            this.y = y;
            this.width = width;
            this.height = height;
        }

        /// <summary>
        /// Creates an element with a hyperlink.
        /// </summary>
        /// <param name="x"> the x position of the clip </param>
        /// <param name="y"> the y position of the clip </param>
        /// <param name="width"> the width of the clip </param>
        /// <param name="height"> the height of the clip </param>
        /// <param name="type">The linktype </param>
        /// <param name="url">the url </param>
        /// <param name="toolTip">the tooltip</param>
        public PageClip(int x, int y, int width, int height, LinkType type, Uri url, string toolTip)
            : this(x, y, width, height)
        {
            this.LinkType = type;
            this.Url = url;
            if (toolTip == null)
            {
                try
                {
                    this.toolTip = HttpUtility.UrlDecode(url.ToString(), Encoding.UTF8).ToLower();
                }
                catch (Exception)
                {
                    // Than it will not be decoded, which should never happen
                    this.toolTip = url.OriginalString;
                }
            }
            else
            {
                this.toolTip = toolTip;
            }
        }

        /// <summary>
        /// Creates an element with an on-demand subreport.
        /// </summary>
        /// <param name="x"> the x position of the clip </param>
        /// <param name="y"> the y position of the clip </param>
        /// <param name="type"> </param>
        /// <param name="url"> </param>
        /// <param name="subreport"> </param>
        /// <param name="width"> the width of the clip </param>
        /// <param name="height"> the height of the clip </param>
        public PageClip(int x, int y, int width, int height, LinkType type, string url, string subreport)
            : this(x, y, width, height)
        {
            this.LinkType = type;
            this.SubReportURL = url;
            toolTip = tabName = subreport;
        }

        /// <summary>
        /// Creates an element with a tool tip.
        /// </summary>
        /// <param name="x"> the x position of the clip </param>
        /// <param name="y"> the y position of the clip </param>
        /// <param name="width"> the width of the clip </param>
        /// <param name="height"> the height of the clip </param>
        /// <param name="toolTip"> </param>
        public PageClip(int x, int y, int width, int height, string toolTip)
            : this(x, y, width, height)
        {
            this.toolTip = toolTip;
            this.LinkType = Data.LinkType.ToolTip;
        }

        /// <summary>
        /// Checks whether the point is within this Page clip </summary>
        /// <param name="x"> x coordiante of the point </param>
        /// <param name="y"> y coordiante of the point </param>
        /// <returns> true if the point is in within the clip</returns>
        public virtual bool Contains(int x, int y)
        {
            return (x >= this.x) && ((x - this.x) < this.width) && (y >= this.y) && ((y - this.y) < this.height);
        }

        /// <summary>
        /// Gets the tool tip.
        /// </summary>
        public string ToolTip
        {
            get
            {
                return toolTip;
            }
            set
            {
                toolTip = value;
            }
        }

        /// <summary>
        /// Compares this instance with the specified instance and
        /// returns true if both have the same locations and dimensions.
        /// </summary>
        /// <param name="other">the other pageclip</param>
        /// <returns>true if both have the same locations and dimensions, otherwise false</returns>
        public bool EqualLocation(PageClip other)
        {
            return x == other.x && y == other.y && width == other.width && height == other.height;
        }
    }
}